# default.py for Streamer
# Features:
# - Auto-detect LAN vs Tailscale IP
# - Add share as Kodi source (JSON-RPC)
# - Speed test (latency to SMB port)
# - Download for offline (best-effort copy with progress)

import sys
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import urllib.parse
import subprocess
import socket
import time
import json
import os
import shutil
import tempfile
import urllib.request

addon = xbmcaddon.Addon(id='plugin.video.streamer')
addonname = addon.getAddonInfo('name') or 'Streamer'
handle = int(sys.argv[1])

# ---------- Helpers ----------

def get_setting(key, default=''):
    # xbmc Addon.getSetting returns str or empty string
    val = addon.getSetting(key)
    return val if val else default

def build_smb_url(ip, share):
    share = share.strip().strip('/')
    return "smb://{}/{}".format(ip.strip(), urllib.parse.quote(share))

def ping_host(ip, timeout_ms=800):
    """Ping the host once. Returns True if reachable. Uses system ping."""
    if not ip:
        return False
    try:
        timeout = max(1, int(timeout_ms/1000))
        result = subprocess.run(["ping", "-c", "1", "-W", str(timeout), ip],
                                stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        return result.returncode == 0
    except Exception:
        return False

def measure_latency(ip, port=445, timeout=2.0):
    """Measure TCP connect latency to ip:port in ms. Returns ms or None."""
    if not ip:
        return None
    try:
        start = time.time()
        s = socket.create_connection((ip, port), timeout=timeout)
        s.close()
        return (time.time() - start) * 1000.0
    except Exception:
        return None

def select_best_ip():
    lan_ip = get_setting('lan_ip')
    tailscale_ip = get_setting('tailscale_ip')
    prefer_tailscale = get_setting('prefer_tailscale') == 'true'

    if prefer_tailscale and tailscale_ip and ping_host(tailscale_ip):
        return tailscale_ip

    if lan_ip and ping_host(lan_ip):
        return lan_ip

    if tailscale_ip and ping_host(tailscale_ip):
        return tailscale_ip

    return lan_ip or tailscale_ip or ''

# ---------- Kodi source (JSON-RPC) ----------

def add_kodi_source(name, smb_url):
    """
    Try to add a video source to Kodi programmatically.
    Uses Files.AddSource JSON-RPC where available.
    """
    try:
        payload = {
            "jsonrpc": "2.0",
            "id": 1,
            "method": "Files.AddSource",
            "params": {
                "media": "video",
                "source": {"name": name, "path": smb_url}
            }
        }
        resp = xbmc.executeJSONRPC(json.dumps(payload))
        res = json.loads(resp) if resp else {}
        if 'error' in res:
            xbmcgui.Dialog().ok(addonname, "Add source failed\n{}".format(str(res.get('error'))))
            return False
        xbmcgui.Dialog().notification(addonname, "Source added: {}".format(smb_url), xbmcgui.NOTIFICATION_INFO, 3000)
        return True
    except Exception as e:
        xbmcgui.Dialog().ok(addonname, "Add source failed\n{}".format(str(e)))
        return False

# ---------- Speed test ----------

def speed_test_for_share(ip, share):
    """Measure latency to SMB port and present a dialog with results."""
    if not ip:
        xbmcgui.Dialog().ok(addonname, "No IP configured\nSet LAN/Tailscale IP in add-on settings.")
        return

    dlg = xbmcgui.DialogProgress()
    dlg.create(addonname, "Testing connection", "Contacting server...")
    xbmc.sleep(300)
    dlg.update(10, "Checking SMB TCP port...")
    ms = measure_latency(ip, port=445, timeout=2.0)
    xbmc.sleep(200)
    if ms is None:
        dlg.update(100, "No response on SMB port (445).", "")
        xbmc.sleep(800)
        dlg.close()
        xbmcgui.Dialog().ok(addonname, "Speed test result\nNo response from {} on port 445.".format(ip))
    else:
        dlg.update(100, "Latency: {:.0f} ms".format(ms), "")
        xbmc.sleep(700)
        dlg.close()
        xbmcgui.Dialog().notification(addonname, "Latency: {:.0f} ms".format(ms), xbmcgui.NOTIFICATION_INFO, 3000)

# ---------- Download for offline (best-effort) ----------

def safe_filename(name):
    return "".join(c for c in name if c.isalnum() or c in " .-_()").strip()

def list_smb_directory(smb_url):
    """Return tuple (dirs, files) using Files.GetDirectory JSON-RPC (best-effort)."""
    try:
        payload = {
            "jsonrpc": "2.0",
            "id": 1,
            "method": "Files.GetDirectory",
            "params": {
                "directory": smb_url,
                "media": "files"
            }
        }
        resp = xbmc.executeJSONRPC(json.dumps(payload))
        res = json.loads(resp) if resp else {}
        dirs = []
        files = []
        if 'result' in res and 'files' in res['result']:
            for item in res['result']['files']:
                if item.get('filetype') == 'directory':
                    dirs.append(item.get('label') or item.get('file'))
                else:
                    files.append(item.get('label') or item.get('file'))
            return dirs, files
    except Exception:
        pass
    try:
        listing = xbmc.listdir(smb_url)
        return listing[0], listing[1]
    except Exception:
        return [], []

def download_file_via_url(remote_url, local_path, dialog=None):
    """Try to download using urllib. Returns True on success."""
    try:
        with urllib.request.urlopen(remote_url, timeout=30) as resp:
            total = resp.getheader('Content-Length')
            total = int(total) if total and total.isdigit() else None
            with open(local_path, 'wb') as out:
                chunk_size = 1024 * 64
                written = 0
                while True:
                    chunk = resp.read(chunk_size)
                    if not chunk:
                        break
                    out.write(chunk)
                    written += len(chunk)
                    if dialog and total:
                        perc = int(written * 100 / total)
                        dialog.update(20 + int(perc * 0.8), "Downloading...", "{}%".format(perc))
        return True
    except Exception as e:
        xbmc.log(f"{addonname}: urllib download failed: {repr(e)}", xbmc.LOGERROR)
        return False

def download_for_offline(smb_url, share_name):
    dlg = xbmcgui.DialogProgress()
    dlg.create(addonname, "Preparing offline download", "Checking remote folder...")
    xbmc.sleep(300)

    dirs, files = list_smb_directory(smb_url)
    if not files and not dirs:
        dlg.close()
        xbmcgui.Dialog().ok(addonname, "Cannot list share\nKodi could not list the SMB folder. Make sure the share is reachable and credentials are valid.")
        return

    total_files = len(files)
    choice = xbmcgui.Dialog().yesno(addonname, "Offline cache", "Found {} files in {}. Download to device?".format(total_files, share_name), yeslabel="Download", nolabel="Cancel")
    if not choice:
        dlg.close()
        return

    dl_path = get_setting('download_path')
    if not dl_path:
        local_root = xbmc.translatePath(os.path.join('special://profile', f'{addonname}Offline'))
    else:
        local_root = xbmc.translatePath(dl_path)
    local_dest = os.path.join(local_root, safe_filename(share_name))
    os.makedirs(local_dest, exist_ok=True)

    dlg.update(5, "Starting downloads...", "")
    xbmc.sleep(200)

    idx = 0
    for f in files:
        idx += 1
        remote_file = smb_url.rstrip('/') + '/' + urllib.parse.quote(f)
        local_file = os.path.join(local_dest, f)
        dlg.update(10, "Downloading {} ({}/{})".format(f, idx, total_files), "")
        ok = download_file_via_url(remote_file, local_file, dialog=dlg)
        if not ok:
            try:
                dlg.update(30, "Attempting filesystem copy...", "")
                shutil.copy(remote_file, local_file)
                ok = True
            except Exception as e:
                xbmc.log(f"{addonname}: filesystem copy failed: {repr(e)}", xbmc.LOGERROR)
                ok = False

        if not ok:
            xbmcgui.Dialog().ok(addonname, "Download failed\nCould not download {}.\nSuggestion:\n- Mount the SMB share on the device (CIFS)\n- Use an Android device with SMB support\n- Copy files via PC then transfer.".format(f))
            dlg.close()
            return

    dlg.update(100, "Download complete", "")
    xbmc.sleep(500)
    dlg.close()
    xbmcgui.Dialog().notification(addonname, "Offline cache ready: {}".format(local_dest), xbmcgui.NOTIFICATION_INFO, 4000)

# ---------- UI routing ----------

def list_shares():
    lan_ip = get_setting('lan_ip')
    tailscale_ip = get_setting('tailscale_ip')
    shares_raw = get_setting('shares', 'MOVIES,SHOWS')
    shares = [s.strip() for s in shares_raw.split(',') if s.strip()]

    active_ip = select_best_ip()

    if not active_ip:
        xbmcgui.Dialog().ok(addonname, "No server IP configured\nPlease set LAN IP or Tailscale IP in add-on settings.")
        return

    xbmcplugin.setPluginCategory(handle, addonname)
    xbmcplugin.setContent(handle, 'videos')

    for share in shares:
        smb_url = build_smb_url(active_ip, share)
        li = xbmcgui.ListItem(label=share)
        li.setProperty('IsPlayable', 'false')

        ctx = []
        if lan_ip:
            ctx.append(('Open via LAN IP', 'RunPlugin(plugin://plugin.video.streamer/?action=open&ip=%s&share=%s)' % (urllib.parse.quote(lan_ip), urllib.parse.quote(share))))
        if tailscale_ip:
            ctx.append(('Open via Tailscale IP', 'RunPlugin(plugin://plugin.video.streamer/?action=open&ip=%s&share=%s)' % (urllib.parse.quote(tailscale_ip), urllib.parse.quote(share))))
        ctx.append(('Copy SMB path', 'RunPlugin(plugin://plugin.video.streamer/?action=copy&ip=%s&share=%s)' % (urllib.parse.quote(active_ip), urllib.parse.quote(share))))
        ctx.append(('Add as Kodi source', 'RunPlugin(plugin://plugin.video.streamer/?action=addsource&ip=%s&share=%s)' % (urllib.parse.quote(active_ip), urllib.parse.quote(share))))
        ctx.append(('Test speed (latency)', 'RunPlugin(plugin://plugin.video.streamer/?action=speed&ip=%s&share=%s)' % (urllib.parse.quote(active_ip), urllib.parse.quote(share))))
        ctx.append(('Download for offline', 'RunPlugin(plugin://plugin.video.streamer/?action=download&ip=%s&share=%s)' % (urllib.parse.quote(active_ip), urllib.parse.quote(share))))

        li.addContextMenuItems(ctx)
        xbmcplugin.addDirectoryItem(handle=handle, url=smb_url, listitem=li, isFolder=True)

    xbmcplugin.endOfDirectory(handle)

def open_share(params):
    ip = urllib.parse.unquote(params.get('ip', ''))
    share = urllib.parse.unquote(params.get('share', ''))
    if not ip:
        xbmcgui.Dialog().ok(addonname, "No IP\nSet LAN or Tailscale IP in settings.")
        return
    smb_url = build_smb_url(ip, share)
    xbmc.executebuiltin('ActivateWindow(Videos,%s,return)' % smb_url)

def copy_share(params):
    ip = urllib.parse.unquote(params.get('ip', ''))
    share = urllib.parse.unquote(params.get('share', ''))
    smb_url = build_smb_url(ip, share)
    xbmc.executebuiltin('Clipboard(%s)' % smb_url)
    xbmcgui.Dialog().notification(addonname, "SMB path copied: {}".format(smb_url), xbmcgui.NOTIFICATION_INFO, 3000)

def add_source_action(params):
    ip = urllib.parse.unquote(params.get('ip', ''))
    share = urllib.parse.unquote(params.get('share', ''))
    smb_url = build_smb_url(ip, share)
    ok = add_kodi_source(share, smb_url)
    if ok and get_setting('auto_add_source') in ('true', True, '1'):
        xbmc.executebuiltin('Container.Update(%s)' % smb_url)

def speed_action(params):
    ip = urllib.parse.unquote(params.get('ip', ''))
    speed_test_for_share(ip, params.get('share', ''))

def download_action(params):
    ip = urllib.parse.unquote(params.get('ip', ''))
    share = urllib.parse.unquote(params.get('share', ''))
    smb_url = build_smb_url(ip, share)
    download_for_offline(smb_url, share)

def router_params(query):
    params = {}
    if query:
        for kv in query.split('&'):
            if '=' in kv and kv.count('=') >= 1:
                k, v = kv.split('=', 1)
                params[k] = v
    return params

def router():
    if len(sys.argv) > 2:
        query = sys.argv[2].lstrip('?')
        params = router_params(query)
        action = params.get('action', '')
        if action == 'open':
            open_share(params)
        elif action == 'copy':
            copy_share(params)
        elif action == 'addsource':
            add_source_action(params)
        elif action == 'speed':
            speed_action(params)
        elif action == 'download':
            download_action(params)
        else:
            list_shares()
    else:
        list_shares()

if __name__ == '__main__':
    router()
